import os
import shutil
import tempfile
import unittest

import pytest
from cytoolz import assoc, assoc_in, get, get_in

import atxmanager as manager
import atxmanager.utils as utils
import atxpylib.comm as atxcomm


# import logging
# logging.basicConfig(level='DEBUG')


# TODO: find a better name and move to some common place
def filter_by_key(d, keys):
    return {k: v for k, v in d.items() if k in keys}


def batch_evidence1_from_batch_request(br):
    return {
        "Date": br["Date"],
        "Time": br["Time"],
        "materials": [filter_by_key(i, ("_k_orig", "Weight", "Humidity")) for i in br["materials"]],
    }


def dlg_data_helper(d, idx, continuous_mode=0, uneven_batches=0):
    op = d["order_params"][d["queue"][idx]]
    if op.get("batch_volume") is not None:  # TODO: does this correctly simulate what happens in gui?
        return {
            "batch_volume": op["batch_volume"],
            "batch_volume_last": op["batch_volume_last"],
            "batch_count": op["batch_count"],
        }
    stru = utils.load_structure_helper(d)
    mixer_volume = utils.get_mixer_volume(stru)
    batch_volume_limit_override = None
    batch_volume_limit = utils.calc_batch_volume_limit(op["_recipe"], mixer_volume, batch_volume_limit_override)
    return utils.calc_batch_volume_etc(op["_order"]["Volume"], continuous_mode, uneven_batches, batch_volume_limit)


class CommonManagerTestCase:
    def setUp(self):
        self.mem = {
            "MEMSH_MA_RecycledWater_disabled": 0,
            "MEMSH_G_Operation": 1,
            "MEMSH_G_Stop_After_Batch": b"",
        }
        self.manager_ini_fn = "manager.ini"
        self.base_ini_fn = "tests/data/base.ini"
        self.parameters_ini_fn = "tests/data/parameters.ini"
        self.settings_ini_fn = "tests/data/settings.ini"
        self.placement_ini_fn = "tests/data/placemnt.ini"
        self.t = 0

        self.tmpdir = tempfile.mkdtemp()
        self.dis_man_dir = f"{self.tmpdir}/dis_man"
        self.man_dis_dir = f"{self.tmpdir}/man_dis"

        os.mkdir(self.dis_man_dir)
        os.mkdir(self.man_dis_dir)
        os.mkdir(f"{self.dis_man_dir}/archive")

    def tearDown(self):
        shutil.rmtree(self.tmpdir)

    def test_save_and_load_state(self):
        fn = "/tmp/manager_test_state.json"  # TODO: hard-coded path
        manager.manager_save_state(self.d, fn)
        manager.manager_load_state(fn)
        # TODO: actually test something

    def test_reset_channels(self):
        manager.manager_reset_channels(self.d)

    def test_auto_load_order(self):
        self.d, self.t = manager.manager_tick(self.d, self.t, self.mem), self.t + 10
        assert self.d["queue"] == []
        shutil.copy("tests/data/order1.ord", self.dis_man_dir)
        # TODO: solve this double shit
        self.d, self.t = manager.manager_tick(self.d, self.t, self.mem), self.t + 10
        self.d, self.t = manager.manager_tick(self.d, self.t, self.mem), self.t + 10
        assert self.d["queue"] == ["order1"]
        assert os.path.isfile(f"{self.man_dis_dir}/order1.acc")
        os.remove(f"{self.man_dis_dir}/order1.acc")
        assert os.path.isfile(f"{self.dis_man_dir}/archive/order1.ord")

    def test_delete_order_from_inside(self):
        self.test_auto_load_order()
        self.d, _ = manager.manager_delete_order(self.d, self.d["queue"][0])
        self.d, self.t = manager.manager_tick(self.d, self.t, self.mem), self.t + 10
        assert self.d["queue"] == []
        assert os.path.isfile(f"{self.man_dis_dir}/order1.ddm")
        assert os.listdir(self.channels["0"]["path_to"]) == []

    def test_delete_order_from_outside(self):
        self.test_auto_load_order()
        shutil.copy("tests/data/order_xxx.DDD", self.dis_man_dir)
        # TODO: solve this double shit
        self.d, self.t = manager.manager_tick(self.d, self.t, self.mem), self.t + 10
        self.d, self.t = manager.manager_tick(self.d, self.t, self.mem), self.t + 10
        assert self.d["queue"] == []
        assert os.path.isfile(f"{self.man_dis_dir}/order_xxx.dac")
        assert os.listdir(self.channels["0"]["path_to"]) == []

    def test_delete_order_from_outside2(self):
        shutil.copy("tests/data/20160411151325_01327.ord", self.dis_man_dir)
        # TODO: solve this double shit
        self.d, self.t = manager.manager_tick(self.d, self.t, self.mem), self.t + 10
        self.d, self.t = manager.manager_tick(self.d, self.t, self.mem), self.t + 10
        assert self.d["queue"] == ["20160411151325_01327"]
        shutil.copy("tests/data/20160411151559_01327.DDD", self.dis_man_dir)
        # TODO: solve this double shit
        self.d, self.t = manager.manager_tick(self.d, self.t, self.mem), self.t + 10
        self.d, self.t = manager.manager_tick(self.d, self.t, self.mem), self.t + 10
        assert self.d["queue"] == []
        assert os.path.isfile(f"{self.man_dis_dir}/20160411151559_01327.dac")
        assert os.listdir(self.channels["0"]["path_to"]) == []


class SingleChannelManagerTestCase(CommonManagerTestCase):
    def setUp(self):
        CommonManagerTestCase.setUp(self)

        self.channels = {"0": manager.new_channel_control(f"{self.tmpdir}/man_ctr", f"{self.tmpdir}/ctr_man")}

        for v in self.channels.values():
            os.mkdir(v["path_to"])
            os.mkdir(v["path_from"])
            os.mkdir(f"{v['path_from']}/archive")


class MultipleChannelsManagerTestCase(CommonManagerTestCase):
    def setUp(self):
        CommonManagerTestCase.setUp(self)
        self.channels = {}
        self.channels = {str(i): manager.new_channel_control(f"{self.tmpdir}/man_ctr{i}", f"{self.tmpdir}/ctr_man{i}") for i in range(2)}
        for v in self.channels.values():
            os.mkdir(v["path_to"])
            os.mkdir(v["path_from"])
            os.mkdir(f"{v['path_from']}/archive")


class SingleChannelBatchModeManagerTestCase(SingleChannelManagerTestCase, unittest.TestCase):
    def setUp(self):
        SingleChannelManagerTestCase.setUp(self)

        self.d = {
            "manager_ini_fn": self.manager_ini_fn,
            "base_ini_fn": self.base_ini_fn,
            "parameters_ini_fn": self.parameters_ini_fn,
            "settings_ini_fn": self.settings_ini_fn,
            "placement_ini_fn": self.placement_ini_fn,
            "dis_man_dir": self.dis_man_dir,
            "man_dis_dir": self.man_dis_dir,
            "channels_control": self.channels,
            "continuous_mode_enabled": 0,
            "queue": [],
            "disallow_multiple_cements": True,
        }
        self.d = manager.manager_load_structure(self.d)

    def test_produce(self):
        self.test_auto_load_order()

        assert len(self.d["queue"]) == 1
        dlg_data = dlg_data_helper(self.d, 0)
        self.d = manager.manager_add_to_produce(self.d, self.d["queue"][0], dlg_data)
        self.d, self.t = manager.manager_tick(self.d, self.t, self.mem), self.t + 10  # TODO: this should not be needed
        assert len(self.d["queue"]) == 1
        assert os.path.isfile(f"{self.man_dis_dir}/order1.std")

        path_to_ = self.channels["0"]["path_to"]
        path_from_ = self.channels["0"]["path_from"]
        for seq in range(1, 11):
            fn_brq = f"{path_to_}/order1_{seq:02d}.brq"
            # assert os.listdir(path_to_) == [f"order1_{seq:02d}.brq"]
            assert os.path.isfile(fn_brq), f"seq: {seq}"
            br = atxcomm.batch_request_from_ini_f(fn_brq)
            assert br["Sequence"] == seq
            be1 = batch_evidence1_from_batch_request(br)
            fn_be1 = f"{path_from_}/order1_{seq:02d}.be1"
            os.rename(fn_brq, fn_be1)
            atxcomm.batch_evidence1_to_ini_f(be1, fn_be1)
            fn_be2 = f"{path_from_}/order1_{seq:02d}.be2"
            shutil.copy(fn_be1, f"{fn_be2}_")
            be2 = {}
            atxcomm.batch_evidence2_to_ini_f(be2, f"{fn_be2}_")
            # TODO: solve this double shit
            self.d, self.t = manager.manager_tick(self.d, self.t, self.mem), self.t + 10
            self.d, self.t = manager.manager_tick(self.d, self.t, self.mem), self.t + 10
            os.rename(f"{fn_be2}_", fn_be2)
            # TODO: solve this double shit
            self.d, self.t = manager.manager_tick(self.d, self.t, self.mem), self.t + 10
            self.d, self.t = manager.manager_tick(self.d, self.t, self.mem), self.t + 10

        assert self.d["queue"] == []
        assert os.listdir(path_to_) == []

    # TODO: test also huge/non-sensical sequence numbers
    def test_resume_order(self):
        self.test_auto_load_order()

        op = self.d["order_params"][self.d["queue"][0]]
        op["batches_sent"] = [1, 2, 3, 4]  # TODO: test all combinations
        op["batches_dosed"] = [1, 2, 3, 4]  # TODO: test all combinations
        op["batches_completed"] = [1, 2, 3, 4]  # TODO: test all combinations
        dlg_data = dlg_data_helper(self.d, 0)
        self.d = manager.manager_add_to_produce(self.d, self.d["queue"][0], dlg_data)
        self.d, self.t = manager.manager_tick(self.d, self.t, self.mem), self.t + 10  # TODO: this should not be needed

        path_to_ = self.channels["0"]["path_to"]
        path_from_ = self.channels["0"]["path_from"]
        for seq in range(5, 11):
            fn_brq = f"{path_to_}/order1_{seq:02d}.brq"
            assert os.path.isfile(fn_brq), f"seq: {seq}"
            br = atxcomm.batch_request_from_ini_f(fn_brq)
            assert br["Sequence"] == seq
            be1 = batch_evidence1_from_batch_request(br)
            fn_be1 = f"{path_from_}/order1_{seq:02d}.be1"
            os.rename(fn_brq, fn_be1)
            atxcomm.batch_evidence1_to_ini_f(be1, fn_be1)
            # TODO: actually save something to be2 file
            fn_be2 = f"{path_from_}/order1_{seq:02d}.be2"
            shutil.copy(fn_be1, f"{fn_be2}_")
            be2 = {}
            atxcomm.batch_evidence2_to_ini_f(be2, f"{fn_be2}_")
            # TODO: solve this double shit
            self.d, self.t = manager.manager_tick(self.d, self.t, self.mem), self.t + 10
            self.d, self.t = manager.manager_tick(self.d, self.t, self.mem), self.t + 10
            os.rename(f"{fn_be2}_", fn_be2)
            # TODO: solve this double shit
            self.d, self.t = manager.manager_tick(self.d, self.t, self.mem), self.t + 10
            self.d, self.t = manager.manager_tick(self.d, self.t, self.mem), self.t + 10

        assert self.d["queue"] == []
        assert not os.path.isfile(f"{path_to_}/order1_11.brq")
        assert not os.path.isfile(f"{self.man_dis_dir}/order1.ddm")


class MultipleChannelsBatchModeManagerTestCase(MultipleChannelsManagerTestCase, unittest.TestCase):
    def setUp(self):
        MultipleChannelsManagerTestCase.setUp(self)

        self.d = {
            "manager_ini_fn": self.manager_ini_fn,
            "base_ini_fn": self.base_ini_fn,
            "parameters_ini_fn": self.parameters_ini_fn,
            "settings_ini_fn": self.settings_ini_fn,
            "placement_ini_fn": self.placement_ini_fn,
            "dis_man_dir": self.dis_man_dir,
            "man_dis_dir": self.man_dis_dir,
            "channels_control": self.channels,
            "continuous_mode_enabled": 0,
            "queue": [],
            "disallow_multiple_cements": True,
        }
        self.d = manager.manager_load_structure(self.d)

    def test_produce(self):
        self.test_auto_load_order()

        assert len(self.d["queue"]) == 1
        dlg_data = dlg_data_helper(self.d, 0)
        self.d = manager.manager_add_to_produce(self.d, self.d["queue"][0], dlg_data)
        self.d, self.t = manager.manager_tick(self.d, self.t, self.mem), self.t + 10  # TODO: this should not be needed
        assert len(self.d["queue"]) == 1
        assert os.path.isfile(f"{self.man_dis_dir}/order1.std")

        while self.d["queue"]:
            something_found = False
            for v in self.channels.values():
                if not os.listdir(v["path_to"]):
                    continue
                something_found = True
                fn = os.listdir(v["path_to"])[0]
                ffn_brq = f"{v['path_to']}/{fn}"
                fn_be1 = fn.replace(".brq", ".be1")
                fn_be2 = fn.replace(".brq", ".be2")
                ffn_be1 = f"{v['path_from']}/{fn_be1}"
                ffn_be2 = f"{v['path_from']}/{fn_be2}"
                br = atxcomm.batch_request_from_ini_f(ffn_brq)
                be1 = batch_evidence1_from_batch_request(br)
                os.rename(ffn_brq, ffn_be1)
                atxcomm.batch_evidence1_to_ini_f(be1, ffn_be1)
                shutil.copy(ffn_be1, ffn_be2)
                # TODO: actually save something to be2 file
                be2 = {}
                atxcomm.batch_evidence2_to_ini_f(be2, ffn_be2)
            assert something_found
            self.d, self.t = manager.manager_tick(self.d, self.t, self.mem), self.t + 10
            self.d, self.t = manager.manager_tick(self.d, self.t, self.mem), self.t + 10

        assert self.d["queue"] == []
        for v in self.channels.values():
            assert os.listdir(v["path_to"]) == []

    # TODO: test also huge/non-sensical sequence numbers
    def test_resume_order(self):
        self.test_auto_load_order()

        op = self.d["order_params"][self.d["queue"][0]]
        op["batches_sent"] = [1, 2, 3, 4]  # TODO: test all combinations
        op["batches_dosed"] = [1, 2, 3, 4]  # TODO: test all combinations
        op["batches_completed"] = [1, 2, 3, 4]  # TODO: test all combinations

        assert len(self.d["queue"]) == 1
        dlg_data = dlg_data_helper(self.d, 0)
        self.d = manager.manager_add_to_produce(self.d, self.d["queue"][0], dlg_data)
        self.d, self.t = manager.manager_tick(self.d, self.t, self.mem), self.t + 10  # TODO: this should not be needed
        assert len(self.d["queue"]) == 1
        assert os.path.isfile(f"{self.man_dis_dir}/order1.std")

        while self.d["queue"]:
            something_found = False
            for v in self.channels.values():
                if not os.listdir(v["path_to"]):
                    continue
                something_found = True
                fn = os.listdir(v["path_to"])[0]
                ffn_brq = f"{v['path_to']}/{fn}"
                br = atxcomm.batch_request_from_ini_f(ffn_brq)
                be1 = batch_evidence1_from_batch_request(br)
                fn_be1 = fn.replace(".brq", ".be1")
                ffn_be1 = f"{v['path_from']}/{fn_be1}"
                os.rename(ffn_brq, ffn_be1)
                atxcomm.batch_evidence1_to_ini_f(be1, ffn_be1)
                fn_be2 = fn.replace(".brq", ".be2")
                ffn_be2 = f"{v['path_from']}/{fn_be2}"
                shutil.copy(ffn_be1, ffn_be2)
                # TODO: actually save something to be2 file
                be2 = {}
                atxcomm.batch_evidence2_to_ini_f(be2, ffn_be2)
            assert something_found
            self.d, self.t = manager.manager_tick(self.d, self.t, self.mem), self.t + 10
            self.d, self.t = manager.manager_tick(self.d, self.t, self.mem), self.t + 10

        assert self.d["queue"] == []


class ContinuousModeManagerTestCase(SingleChannelManagerTestCase, unittest.TestCase):
    def setUp(self):
        SingleChannelManagerTestCase.setUp(self)

        self.d = {
            "manager_ini_fn": self.manager_ini_fn,
            "base_ini_fn": self.base_ini_fn,
            "parameters_ini_fn": self.parameters_ini_fn,
            "settings_ini_fn": self.settings_ini_fn,
            "placement_ini_fn": self.placement_ini_fn,
            "dis_man_dir": self.dis_man_dir,
            "man_dis_dir": self.man_dis_dir,
            "channels_control": self.channels,
            "continuous_mode_enabled": 1,
            "queue": [],
            "disallow_multiple_cements": True,
        }
        self.d = manager.manager_load_structure(self.d)

    def test_produce(self):
        self.test_auto_load_order()

        path_to_ = self.channels["0"]["path_to"]
        path_from_ = self.channels["0"]["path_from"]

        assert len(self.d["queue"]) == 1
        dlg_data = dlg_data_helper(self.d, 0, continuous_mode=1)
        self.d = manager.manager_add_to_produce(self.d, self.d["queue"][0], dlg_data)
        self.d, self.t = manager.manager_tick(self.d, self.t, self.mem), self.t + 10  # TODO: this should not be needed
        assert len(self.d["queue"]) == 1
        assert os.path.isfile(f"{self.man_dis_dir}/order1.std")
        assert os.path.isfile(f"{path_to_}/order1_01.brq")

        os.rename(f"{path_to_}/order1_01.brq", f"{path_from_}/order1_01.be1")

        br = atxcomm.batch_request_from_ini_f(f"{path_from_}/order1_01.be1")
        assert br["Sequence"] == 1

        be1 = batch_evidence1_from_batch_request(br)
        fn_be1 = f"{path_from_}/order1_01.be1"
        atxcomm.batch_evidence1_to_ini_f(be1, fn_be1)
        fn_be2 = f"{path_from_}/order1_01.be2"
        shutil.copy(fn_be1, f"{fn_be2}_")
        be2 = {}
        atxcomm.batch_evidence2_to_ini_f(be2, f"{fn_be2}_")
        # TODO: solve this double shit
        self.d, self.t = manager.manager_tick(self.d, self.t, self.mem), self.t + 10
        self.d, self.t = manager.manager_tick(self.d, self.t, self.mem), self.t + 10
        os.rename(f"{fn_be2}_", fn_be2)
        # TODO: solve this double shit
        self.d, self.t = manager.manager_tick(self.d, self.t, self.mem), self.t + 10
        self.d, self.t = manager.manager_tick(self.d, self.t, self.mem), self.t + 10

        assert self.d["queue"] == []
        assert os.listdir(self.channels["0"]["path_to"]) == []
        assert not os.path.isfile(f"{self.man_dis_dir}/order1.ddm")


def gen_recipes():
    yield {
        "materials": [],
        "ACCAWR": 1,
    }
    yield atxcomm.recipe_from_ini_f("tests/data/order1.ord")


@pytest.fixture(params=gen_recipes())
def recipe(request):
    return request.param


@pytest.fixture(params=[False, True])
def legacy_water(request):
    return request.param


@pytest.fixture(params=[False, True])
def continuous_mode_enabled(request):
    return request.param


@pytest.fixture
def manager_(request, recipe, legacy_water, continuous_mode_enabled):
    return {
        "queue": ["aaa"],
        "order_params": {
            "aaa": {
                "_recipe": recipe,
                "_order": {
                    "VehicleIdentificationNumber": "abc",
                    "Volume": 10,
                },
                "batch_correction": 1,
            }
        },
        "display_mode": "recipe",
        "continuous_mode_enabled": continuous_mode_enabled,
        "legacy_water": legacy_water,
    }


@pytest.fixture
def manager__(tmpdir):
    dis_man_dir = f"{tmpdir}/dis_man"
    os.mkdir(dis_man_dir)
    os.mkdir(f"{dis_man_dir}/archive")
    man_dis_dir = f"{tmpdir}/man_dis"
    os.mkdir(man_dis_dir)
    os.mkdir(f"{man_dis_dir}/archive")
    d = {
        "channels_control": {
            "0": {
                "path_from": f"{tmpdir}/ctr_man",
                "path_to": f"{tmpdir}/man_ctr",
            },
        },
        "dis_man_dir": dis_man_dir,
        "man_dis_dir": man_dis_dir,
        "base_ini_fn": "tests/data/base.ini",
        "parameters_ini_fn": "tests/data/parameters.ini",
        "settings_ini_fn": "tests/data/settings.ini",
        "placement_ini_fn": "tests/data/placemnt.ini",
        "disallow_multiple_cements": True,
    }
    d = manager.manager_load_structure(d)
    for v in d["channels_control"].values():
        os.mkdir(v["path_to"])
        os.mkdir(v["path_from"])
        os.mkdir(f"{v['path_from']}/archive")
    return d


@pytest.fixture
def mem():
    return {
        "MEMSH_MA_RecycledWater_disabled": 0,
        "MEMSH_G_Operation": 1,
        "MEMSH_G_Stop_After_Batch": b"",
    }


@pytest.fixture
def tmpdir():
    t = tempfile.mkdtemp()
    yield t
    shutil.rmtree(t)


def _auto_load_order(d, t, fn, mem):
    d, t = manager.manager_tick(d, t, mem), t + 10
    assert fn not in d.get("queue", [])
    shutil.copy(f"tests/data/{fn}.ord", d["dis_man_dir"])
    # TODO: solve this double shit
    d, t = manager.manager_tick(d, t, mem), t + 10
    d, t = manager.manager_tick(d, t, mem), t + 10
    assert fn in d.get("queue", [])
    assert fn in d["order_params"]
    assert os.path.isfile(f"{d['man_dis_dir']}/{fn}.acc")
    os.remove(f"{d['man_dis_dir']}/{fn}.acc")
    assert os.path.isfile(f"{d['dis_man_dir']}/archive/{fn}.ord")
    return d, t, mem


def test_manager_produce(manager__, mem):
    d, t = manager__, 0
    d, t, mem = _auto_load_order(d, t, "order1", mem)
    d, t, mem = _auto_load_order(d, t, "order2", mem)
    d, t, mem = _auto_load_order(d, t, "order3", mem)
    assert len(d["queue"]) == 3
    dlg_data = dlg_data_helper(d, 0)
    d = manager.manager_add_to_produce(d, d["queue"][0], dlg_data)
    assert d["order_params"]["order1"]["batch_volume"] == 1
    assert d["order_params"]["order1"]["batch_volume_last"] is None
    dlg_data = dlg_data_helper(d, 1)
    d = manager.manager_add_to_produce(d, d["queue"][1], dlg_data)
    assert d["order_params"]["order2"]["batch_volume"] == 0.95
    assert d["order_params"]["order2"]["batch_volume_last"] is None
    dlg_data = dlg_data_helper(d, 2, uneven_batches=1)
    d = manager.manager_add_to_produce(d, d["queue"][2], dlg_data)
    assert d["order_params"]["order3"]["batch_volume"] == 1
    assert d["order_params"]["order3"]["batch_volume_last"] == 0.5
    d, t = manager.manager_tick(d, t, mem), t + 10  # TODO: this should not be needed
    assert os.path.isfile(f"{d['man_dis_dir']}/order1.std")
    assert not os.path.isfile(f"{d['man_dis_dir']}/order2.std")
    fns_and_seqs = []
    fns_and_seqs.extend([("order1", i) for i in range(1, 11)])
    fns_and_seqs.extend([("order2", i) for i in range(1, 11)])
    fns_and_seqs.extend([("order3", i) for i in range(1, 11)])
    path_to_ = d["channels_control"]["0"]["path_to"]
    path_from_ = d["channels_control"]["0"]["path_from"]
    for fn, seq in fns_and_seqs:
        fn_brq = f"{path_to_}/{fn}_{seq:02d}.brq"
        # assert(os.listdir(path_to_) == [f"{fn}_{seq:02d}.brq"])
        assert os.path.isfile(fn_brq), f"fn: {fn} seq: {seq}"
        br = atxcomm.batch_request_from_ini_f(fn_brq)
        assert br["Sequence"] == seq, f"fn: {fn} seq: {seq}"
        if fn == "order3":
            if seq == 10:
                assert br["Volume"] == 0.5
            else:
                assert br["Volume"] == 1
        be1 = batch_evidence1_from_batch_request(br)
        fn_be1 = f"{path_from_}/{fn}_{seq:02d}.be1"
        os.rename(fn_brq, fn_be1)
        atxcomm.batch_evidence1_to_ini_f(be1, fn_be1)
        fn_be2 = f"{path_from_}/{fn}_{seq:02d}.be2"
        shutil.copy(fn_be1, f"{fn_be2}_")
        be2 = {}
        atxcomm.batch_evidence2_to_ini_f(be2, f"{fn_be2}_")
        # TODO: solve this double shit
        d, t = manager.manager_tick(d, t, mem), t + 10
        d, t = manager.manager_tick(d, t, mem), t + 10
        assert not os.path.isfile(fn_be1), f"fn: {fn} seq: {seq}"
        os.rename(f"{fn_be2}_", fn_be2)
        # TODO: solve this double shit
        d, t = manager.manager_tick(d, t, mem), t + 10
        d, t = manager.manager_tick(d, t, mem), t + 10
        assert not os.path.isfile(fn_be2), f"fn: {fn} seq: {seq}"
    assert os.path.isfile(f"{d['man_dis_dir']}/order2.std")  # TODO: we should check this sooner
    assert d["queue"] == []
    assert os.listdir(path_to_) == []


# TODO: unite with the one below
@pytest.mark.parametrize("kdx_quirk_enabled", [False, True])
def test_manager_produce_and_delete_first_order_half_way(manager__, mem, kdx_quirk_enabled):
    d, t = manager__, 0
    d["quirk_kdx_fake_evidence_files"] = kdx_quirk_enabled
    d, t, mem = _auto_load_order(d, t, "order1", mem)
    d, t, mem = _auto_load_order(d, t, "order2", mem)
    dlg_data = dlg_data_helper(d, 0)
    d = manager.manager_add_to_produce(d, d["queue"][0], dlg_data)
    dlg_data = dlg_data_helper(d, 1)
    d = manager.manager_add_to_produce(d, d["queue"][1], dlg_data)
    d, t = manager.manager_tick(d, t, mem), t + 10  # TODO: this should not be needed
    assert len(d["queue"]) == 2
    man_dis_dir_ = d["man_dis_dir"]
    assert os.path.isfile(f"{man_dis_dir_}/order1.std")
    assert not os.path.isfile(f"{man_dis_dir_}/order2.std")
    fns_and_seqs = []
    fns_and_seqs.extend([("order1", i) for i in range(1, 7)])
    fns_and_seqs.extend([("order2", i) for i in range(1, 11)])
    path_to_ = d["channels_control"]["0"]["path_to"]
    path_from_ = d["channels_control"]["0"]["path_from"]
    for fn, seq in fns_and_seqs:
        fn_brq = f"{path_to_}/{fn}_{seq:02d}.brq"
        # assert(os.listdir(path_to_) == [f"{fn}_{seq:02d}.brq")])
        assert os.path.isfile(fn_brq), f"fn: {fn} seq: {seq}"
        br = atxcomm.batch_request_from_ini_f(fn_brq)
        assert br["Sequence"] >= 1, f"fn: {fn} seq: {seq}"
        if (fn, seq) == ("order1", 6):
            d, deleted = manager.manager_delete_order(d, "order1")
            assert deleted
            assert not os.path.isfile(f"{man_dis_dir_}/{fn}.ddm")
            assert os.path.isfile(f"{man_dis_dir_}/{fn}.int")
            int_ = atxcomm.interruption_from_ini_f(f"{man_dis_dir_}/{fn}.int")
            assert int_["batch_count"] == 10
            assert int_["batches_sent"] == "1,2,3,4,5,6"
            assert int_["batches_dosed"] == "1,2,3,4,5"
            assert int_["batches_completed"] == "1,2,3,4,5"
            assert int_["t"]
            os.remove(f"{man_dis_dir_}/{fn}.int")
            assert os.path.isfile(f"{path_to_}/{fn}.int")
            os.remove(f"{path_to_}/{fn}.int")
            # TODO: solve this double shit
            d, t = manager.manager_tick(d, t, mem), t + 10
            d, t = manager.manager_tick(d, t, mem), t + 10
        be1 = batch_evidence1_from_batch_request(br)
        fn_be1 = f"{path_from_}/{fn}_{seq:02d}.be1"
        os.rename(fn_brq, fn_be1)
        atxcomm.batch_evidence1_to_ini_f(be1, fn_be1)
        fn_be2 = f"{path_from_}/{fn}_{seq:02d}.be2"
        shutil.copy(fn_be1, f"{fn_be2}_")
        be2 = {}
        atxcomm.batch_evidence2_to_ini_f(be2, f"{fn_be2}_")
        # TODO: solve this double shit
        d, t = manager.manager_tick(d, t, mem), t + 10
        d, t = manager.manager_tick(d, t, mem), t + 10
        assert not os.path.isfile(fn_be1), f"fn: {fn} seq: {seq}"
        os.rename(f"{fn_be2}_", fn_be2)
        # TODO: solve this double shit
        d, t = manager.manager_tick(d, t, mem), t + 10
        d, t = manager.manager_tick(d, t, mem), t + 10
        assert not os.path.isfile(fn_be2), f"fn: {fn} seq: {seq}"
    assert os.path.isfile(f"{man_dis_dir_}/order2.std")  # TODO: we should check this sooner
    for fn, seq in fns_and_seqs:
        should_exist = True
        if fn == "order1" and seq > 6 and kdx_quirk_enabled:
            should_exist = False
        if should_exist:
            assert os.path.isfile(f"{man_dis_dir_}/{fn}_{seq:02d}.be1")
            assert os.path.isfile(f"{man_dis_dir_}/{fn}_{seq:02d}.be2")
        else:
            assert not os.path.isfile(f"{man_dis_dir_}/{fn}_{seq:02d}.be1")
            assert not os.path.isfile(f"{man_dis_dir_}/{fn}_{seq:02d}.be2")
    d, t = manager.manager_tick(d, t, mem), t + 10
    assert d["queue"] == []
    assert os.listdir(path_to_) == []


def test_manager_produce_and_delete_from_inside(manager__, mem):
    d, t = manager__, 0
    d, t, mem = _auto_load_order(d, t, "order1", mem)
    dlg_data = dlg_data_helper(d, 0)
    d = manager.manager_add_to_produce(d, d["queue"][0], dlg_data)
    d, t = manager.manager_tick(d, t, mem), t + 10  # TODO: this should not be needed
    assert len(d["queue"]) == 1
    man_dis_dir_ = d["man_dis_dir"]
    assert os.path.isfile(f"{man_dis_dir_}/order1.std")
    path_to_ = d["channels_control"]["0"]["path_to"]
    path_from_ = d["channels_control"]["0"]["path_from"]
    seq = 1
    fn_brq = f"{path_to_}/order1_{seq:02d}.brq"
    assert os.path.isfile(fn_brq), f"seq: {seq}"
    d, deleted = manager.manager_delete_order(d, d["queue"][0])
    assert deleted
    assert not os.path.isfile(f"{man_dis_dir_}/order1.ddm")
    assert os.path.isfile(f"{man_dis_dir_}/order1.int")
    assert os.path.isfile(f"{path_to_}/order1.int")
    d, t = manager.manager_tick(d, t, mem), t + 10
    # we used to wait for everything to complete but it led to zombie orders (after power failure, ...)
    # assert len(d['queue']) == 1
    assert d["queue"] == []
    fn_brq_next = f"{path_to_}/order1_{(seq + 1):02d}.brq"
    assert not os.path.isfile(fn_brq_next)
    br = atxcomm.batch_request_from_ini_f(fn_brq)
    be1 = batch_evidence1_from_batch_request(br)
    fn_be1 = f"{path_from_}/order1_{seq:02d}.be1"
    os.rename(fn_brq, fn_be1)
    atxcomm.batch_evidence1_to_ini_f(be1, fn_be1)
    fn_be2 = f"{path_from_}/order1_{seq:02d}.be2"
    shutil.copy(fn_be1, f"{fn_be2}_")
    be2 = {}
    atxcomm.batch_evidence2_to_ini_f(be2, f"{fn_be2}_")
    # TODO: solve this double shit
    d, t = manager.manager_tick(d, t, mem), t + 10
    d, t = manager.manager_tick(d, t, mem), t + 10
    # we used to wait for everything to complete but it led to zombie orders (after power failure, ...)
    # assert len(d['queue']) == 1
    assert d["queue"] == []
    os.rename(f"{fn_be2}_", fn_be2)
    # TODO: solve this double shit
    d, t = manager.manager_tick(d, t + 10, mem), t + 10
    d, t = manager.manager_tick(d, t + 10, mem), t + 10
    assert d["queue"] == []


def test_manager_produce_and_delete_from_outside(manager__, mem):
    d, t = manager__, 0
    d, t, mem = _auto_load_order(d, t, "order1", mem)
    dlg_data = dlg_data_helper(d, 0)
    d = manager.manager_add_to_produce(d, d["queue"][0], dlg_data)
    d, t = manager.manager_tick(d, t, mem), t + 10  # TODO: this should not be needed
    assert len(d["queue"]) == 1
    man_dis_dir_ = d["man_dis_dir"]
    assert os.path.isfile(f"{man_dis_dir_}/order1.std")
    d, deleted = manager.manager_delete_order(d, d["queue"][0], from_outside=True)
    assert not deleted
    assert not os.path.isfile(f"{man_dis_dir_}/order1.ddm")
    assert not os.path.isfile(f"{man_dis_dir_}/order1.int")
    path_to_ = d["channels_control"]["0"]["path_to"]
    assert not os.path.isfile(f"{path_to_}/order1.int")
    d, t = manager.manager_tick(d, t, mem), t + 10
    assert len(d["queue"]) == 1


def test_delete_order_from_outside(manager__, mem):
    d, t = manager__, 0
    d, t, mem = _auto_load_order(d, t, "order1", mem)
    shutil.copy("tests/data/order_xxx.DDD", d["dis_man_dir"])
    # TODO: solve this double shit
    d, t = manager.manager_tick(d, t, mem), t + 10
    d, t = manager.manager_tick(d, t, mem), t + 10
    assert d["queue"] == []
    assert os.path.isfile(f"{d['man_dis_dir']}/order_xxx.dac")
    assert os.listdir(d["channels_control"]["0"]["path_to"]) == []


def test_delete_order(manager__, mem):
    d, t = manager__, 0
    d, t, mem = _auto_load_order(d, t, "order1", mem)
    d, _ = manager.manager_delete_order(d, d["queue"][0])
    man_dis_dir_ = d["man_dis_dir"]
    assert os.path.isfile(f"{man_dis_dir_}/order1.ddm")
    assert not os.path.isfile(f"{man_dis_dir_}/order1.int")
    path_to_ = d["channels_control"]["0"]["path_to"]
    assert not os.path.isfile(f"{path_to_}/order1.int")
    assert os.listdir(d["channels_control"]["0"]["path_to"]) == []
    d, t = manager.manager_tick(d, t, mem), t + 10
    assert d["queue"] == []


def test_skip_order_with_problems(manager__, mem):
    d, t = manager__, 0
    d, t, mem = _auto_load_order(d, t, "order1_unplaced", mem)
    d, t, mem = _auto_load_order(d, t, "order2", mem)
    dlg_data = dlg_data_helper(d, 0)
    d = manager.manager_add_to_produce(d, d["queue"][0], dlg_data)
    dlg_data = dlg_data_helper(d, 1)
    d = manager.manager_add_to_produce(d, d["queue"][1], dlg_data)
    d, t = manager.manager_tick(d, t, mem), t + 10
    seq = 1
    path_to_ = d["channels_control"]["0"]["path_to"]
    fn_brq = f"{path_to_}/order1_unplaced_{seq:02d}.brq"
    assert not os.path.isfile(fn_brq), f"seq: {seq}"
    seq = 1
    fn_brq = f"{path_to_}/order2_{seq:02d}.brq"
    assert not os.path.isfile(fn_brq), f"seq: {seq}"


def test_manager_produce_and_placement_change(manager__, mem):
    d, t = manager__, 0
    d, t, mem = _auto_load_order(d, t, "order1", mem)
    dlg_data = dlg_data_helper(d, 0)
    d = manager.manager_add_to_produce(d, d["queue"][0], dlg_data)
    d, t = manager.manager_tick(d, t, mem), t + 10  # TODO: this should not be needed
    assert len(d["queue"]) == 1
    assert d["order_params"]["order1"].get("batches_dosed", []) == []
    assert d["order_params"]["order1"]["batches_sent"] == [1]
    assert os.path.isfile(f"{d['man_dis_dir']}/order1.std")
    seq = 1
    path_to_ = d["channels_control"]["0"]["path_to"]
    path_from_ = d["channels_control"]["0"]["path_from"]
    fn_brq = f"{path_to_}/order1_{seq:02d}.brq"
    assert os.path.isfile(fn_brq), f"seq: {seq}"
    d = assoc(d, "placement_ini_fn", "tests/data/empty.ini")
    # a call to manager_tick should be here but since we check mtime of placements
    # file, the change may not be registered - so i call load_placements directly
    d = manager.manager_load_placements(d, d["placement_ini_fn"], d.get("_silo_to_bin"))
    br = atxcomm.batch_request_from_ini_f(fn_brq)
    be1 = batch_evidence1_from_batch_request(br)
    fn_be1 = f"{path_from_}/order1_{seq:02d}.be1"
    os.rename(fn_brq, fn_be1)
    atxcomm.batch_evidence1_to_ini_f(be1, fn_be1)
    fn_be2 = f"{path_from_}/order1_{seq:02d}.be2"
    shutil.copy(fn_be1, f"{fn_be2}_")
    be2 = {}
    atxcomm.batch_evidence2_to_ini_f(be2, f"{fn_be2}_")
    # TODO: solve this double shit
    d, t = manager.manager_tick(d, t, mem), t + 10
    d, t = manager.manager_tick(d, t, mem), t + 10
    os.rename(f"{fn_be2}_", fn_be2)
    # TODO: solve this double shit
    d, t = manager.manager_tick(d, t, mem), t + 10
    d, t = manager.manager_tick(d, t, mem), t + 10
    assert d["order_params"]["order1"]["batches_dosed"] == [1]
    assert d["order_params"]["order1"]["batches_sent"] == [1]
    assert d["order_params"]["order1"]["problems"]
    d = assoc(d, "placement_ini_fn", "tests/data/placemnt.ini")
    # a call to manager_tick should be here but since we check mtime of placements
    # file, the change may not be registered - so i call load_placements directly
    d = manager.manager_load_placements(d, d["placement_ini_fn"], d.get("_silo_to_bin"))
    stru = utils.load_structure_helper(d)
    d = assoc(d, "_stru", stru)
    d = manager.manager_fill_problems_and_warnings(d, "order1", d["channels_control"]["0"])
    assert not d["order_params"]["order1"]["problems"]
    d, t = manager.manager_tick(d, t, mem), t + 10
    assert d["order_params"]["order1"]["batches_sent"] == [1, 2]


# TODO: this used to be commented out - is it unfinished or what?
def _test_manager_produce_and_mixer_volume_change_UNUSED(manager__, mem):
    d, t = manager__, 0
    d, t, mem = _auto_load_order(d, t, "order1", mem)
    dlg_data = dlg_data_helper(d, 0)
    d = manager.manager_add_to_produce(d, d["queue"][0], dlg_data)
    d, t = manager.manager_tick(d, t, mem), t + 10  # TODO: this should not be needed
    assert len(d["queue"]) == 1
    assert d["order_params"]["order1"].get("batches_dosed", []) == []
    assert d["order_params"]["order1"]["batches_sent"] == [1]
    assert os.path.isfile(f"{d['man_dis_dir']}/order1.std")
    seq = 1
    path_to_ = d["channels_control"]["0"]["path_to"]
    path_from_ = d["channels_control"]["0"]["path_from"]
    fn_brq = f"{path_to_}/order1_{seq:02d}.brq"
    assert os.path.isfile(fn_brq), f"seq: {seq}"
    br = atxcomm.batch_request_from_ini_f(fn_brq)
    assert get("Volume", br) == 1
    be1 = batch_evidence1_from_batch_request(br)
    fn_be1 = f"{path_from_}/order1_{seq:02d}.be1"
    os.rename(fn_brq, fn_be1)
    atxcomm.batch_evidence1_to_ini_f(be1, fn_be1)
    fn_be2 = f"{path_from_}/order1_{seq:02d}.be2"
    shutil.copy(fn_be1, f"{fn_be2}_")
    be2 = {}
    atxcomm.batch_evidence2_to_ini_f(be2, f"{fn_be2}_")
    d = assoc(d, "parameters_ini_fn", "tests/data/parameters_small_mixer.ini")
    d, t = manager.manager_tick(d, t, mem), t + 10
    os.rename(f"{fn_be2}_", fn_be2)
    # TODO: solve this double shit
    d, t = manager.manager_tick(d, t, mem), t + 10
    d, t = manager.manager_tick(d, t, mem), t + 10
    assert d["order_params"]["order1"]["batches_dosed"] == [1]
    assert d["order_params"]["order1"]["batches_sent"] == [1, 2]
    seq = 2
    fn_brq = f"{path_to_}/order1_{seq:02d}.brq"
    assert os.path.isfile(fn_brq), f"seq: {seq}"
    br = atxcomm.batch_request_from_ini_f(fn_brq)
    assert get("Volume", br) == 1


# TODO: this used to be commented out - is it unfinished or what?
def _test_manager_produce_and_restart_and_mixer_volume_change_UNUSED(manager__, mem):
    d, t = manager__, 0
    d, t, mem = _auto_load_order(d, t, "order1", mem)
    dlg_data = dlg_data_helper(d, 0)
    d = manager.manager_add_to_produce(d, d["queue"][0], dlg_data)
    d, t = manager.manager_tick(d, t, mem), t + 10  # TODO: this should not be needed
    assert len(d["queue"]) == 1
    assert d["order_params"]["order1"].get("batches_dosed", []) == []
    assert d["order_params"]["order1"]["batches_sent"] == [1]
    assert os.path.isfile(f"{d['man_dis_dir']}/order1.std")
    seq = 1
    path_to_ = d["channels_control"]["0"]["path_to"]
    path_from_ = d["channels_control"]["0"]["path_from"]
    fn_brq = f"{path_to_}/order1_{seq:02d}.brq"
    assert os.path.isfile(fn_brq), f"seq: {seq}"
    br = atxcomm.batch_request_from_ini_f(fn_brq)
    assert get("Volume", br) == 1
    be1 = batch_evidence1_from_batch_request(br)
    fn_be1 = f"{path_from_}/order1_{seq:02d}.be1"
    os.rename(fn_brq, fn_be1)
    atxcomm.batch_evidence1_to_ini_f(be1, fn_be1)
    fn_be2 = f"{path_from_}/order1_{seq:02d}.be2"
    shutil.copy(fn_be1, f"{fn_be2}_")
    be2 = {}
    atxcomm.batch_evidence2_to_ini_f(be2, f"{fn_be2}_")
    # the following simulates restart
    d = manager.manager_remove_orders_with_non_existent_files(d)
    d = manager.manager_remove_orders_with_no_more_batches_to_send(d)
    d = manager.manager_reset_order_states(d)
    d = manager.manager_reset_channels(d)
    d = assoc(d, "parameters_ini_fn", "tests/data/parameters_small_mixer.ini")
    dlg_data = dlg_data_helper(d, 0)
    d = manager.manager_add_to_produce(d, d["queue"][0], dlg_data)
    d, t = manager.manager_tick(d, t, mem), t + 10
    os.rename(f"{fn_be2}_", fn_be2)
    # TODO: solve this double shit
    d, t = manager.manager_tick(d, t, mem), t + 10
    d, t = manager.manager_tick(d, t, mem), t + 10
    assert d["order_params"]["order1"]["batches_dosed"] == [1]
    assert d["order_params"]["order1"]["batches_sent"] == [1, 2]
    seq = 2
    fn_brq = f"{path_to_}/order1_{seq:02d}.brq"
    assert os.path.isfile(fn_brq), f"seq: {seq}"
    br = atxcomm.batch_request_from_ini_f(fn_brq)
    assert get("Volume", br) == 1


def test_manager_remove_orders_with_non_existent_files():
    d = {
        "queue": ["aaa"],
        "order_params": {
            "aaa": {
                "ffn": "/non/existent/file",
            }
        },
    }
    d = manager.manager_remove_orders_with_non_existent_files(d)
    assert d["queue"] == []
    assert d["order_params"] == {}


def test_manager_reset_order_states():
    d = {
        "queue": ["aaa"],
        "order_params": {
            "aaa": {
                "ffn": "/non/existent/file",
                "state": "production",
            }
        },
    }
    d = manager.manager_reset_order_states(d)
    assert get_in(["order_params", "aaa", "state"], d) is None
    d = assoc_in(d, ["order_params", "aaa", "state"], "ready")
    d = manager.manager_reset_order_states(d)
    assert get_in(["order_params", "aaa", "state"], d) is None


def test_manager_return_order():
    d = {"queue": ["aa"], "order_params": {"aa": {"state": "ready"}}}
    d = manager.manager_return_order(d, "aa")
    assert get_in(["order_params", "aa", "state"], d) is None
    d = {"queue": ["aa"], "order_params": {"aa": {"state": "ready"}}}
    d = manager.manager_return_order(d, "bb")
    assert get_in(["order_params", "aa", "state"], d) == "ready"


def test_manager_recycled_only_when_recycled_disabled(mem):
    d = {
        "channels_control": {
            "0": {
                "path_from": f"{tmpdir}/ctr_man",
                "path_to": f"{tmpdir}/man_ctr",
            },
        },
        "queue": ["aa"],
        "order_params": {
            "aa": {
                "_recipe": {
                    "materials": [{"Name": "KALOVA", "type": "Water", "Weight": 100}],
                },
            }
        },
        "disallow_multiple_cements": True,
        "structure_hjson_fn": None,
        "base_ini_fn": "tests/data/base.ini",
        "parameters_ini_fn": "tests/data/parameters.ini",
        "settings_ini_fn": "tests/data/settings.ini",
        "placement_ini_fn": "tests/data/placemnt.ini",
    }
    d = manager.manager_load_structure(d)
    d = manager.manager_load_moistures(d, {})  # TODO
    d = manager.manager_load_temperatures(d, {})  # TODO
    d = manager.manager_load_densities(d, {})  # TODO
    d = manager.manager_load_placements(d, d["placement_ini_fn"], d.get("_silo_to_bin"))
    for k, v in d["channels_control"].items():
        d, v = manager.manager_check_channel_control_shared_memory(d, v, mem)
        d = assoc_in(d, ["channels_control", k], v)
    d = manager.manager_fill_problems_and_warnings(d, "aa", d["channels_control"]["0"])
    # TODO: this is unfinished and probably broken - in the end it should test CASE1


def test_manager_send_next_batch_request_all_sent(mem):
    d = {
        "order_params": {
            "aaa": {
                "batch_count": 4,
                "batches_sent": [1, 2, 3, 4],
            },
        },
        "structure_hjson_fn": None,
        "base_ini_fn": "tests/data/base.ini",
        "parameters_ini_fn": "tests/data/parameters.ini",
        "settings_ini_fn": "tests/data/settings.ini",
        "placement_ini_fn": "tests/data/placemnt.ini",
    }
    d = manager.manager_load_structure(d)
    d = manager.manager_load_moistures(d, {})  # TODO
    d = manager.manager_load_placements(d, d["placement_ini_fn"], d.get("_silo_to_bin"))
    fn = "aaa"
    channel = {}
    manager.manager_send_next_batch_request(d, fn, channel, mem)


def test_manager_load_fucked_be2(manager__, mem):
    d, t = manager__, 0
    path_from_ = d["channels_control"]["0"]["path_from"]
    shutil.copy("tests/data/.be2", f"{path_from_}/.be2")
    d, t = manager.manager_tick(d, t + 10, mem), t + 10
    d, t = manager.manager_tick(d, t + 10, mem), t + 10
    assert not os.path.isfile(f"{path_from_}/.be2")
    assert os.path.isfile(f"{path_from_}/archive/.be2")
    assert os.path.isfile(f"{d['man_dis_dir']}/.be2")


# TODO: create a special testing suite for api?
def test_manager_get_fill_data(manager__, mem):
    d, t = manager__, 0
    d, t, mem = _auto_load_order(d, t, "order1", mem)
    fill_data = utils.get_fill_data(d, "order1", {})
    assert "batch_count" in fill_data
    assert utils.get_fill_data(d, "order1", None) == {}
    assert utils.get_fill_data(d, "non_existent", {}) == {}
    assert utils.get_fill_data(d, "non_existent", None) == {}
