"""Peewee migrations -- 159_MAIN.py.

Some examples (model - class or model name)::

    > Model = migrator.orm['table_name']            # Return model in current state by name
    > Model = migrator.ModelClass                   # Return model in current state by name

    > migrator.sql(sql)                             # Run custom SQL
    > migrator.run(func, *args, **kwargs)           # Run python function with the given args
    > migrator.create_model(Model)                  # Create a model (could be used as decorator)
    > migrator.remove_model(model, cascade=True)    # Remove a model
    > migrator.add_fields(model, **fields)          # Add fields to a model
    > migrator.change_fields(model, **fields)       # Change fields
    > migrator.remove_fields(model, *field_names, cascade=True)
    > migrator.rename_field(model, old_field_name, new_field_name)
    > migrator.rename_table(model, new_table_name)
    > migrator.add_index(model, *col_names, unique=False)
    > migrator.add_not_null(model, *field_names)
    > migrator.add_default(model, field_name, default)
    > migrator.add_constraint(model, name, sql)
    > migrator.drop_index(model, *col_names)
    > migrator.drop_not_null(model, *field_names)
    > migrator.drop_constraints(model, *constraints)
"""

import peewee as pw
from peewee_migrate import Migrator


def extract_fns(migrator: Migrator) -> None:
    if not (setup := migrator.orm["setup"].get_or_none()):
        return

    from atxdispatch.dispatch import load_setup
    from atxdispatch import settings

    config = load_setup(settings.CONFIG_FILE)

    if delivery_sheet_template_fn := config.get("delivery_sheet_template_fn", config.get("delivery_sheet_name")):
        setup.delivery_sheet_fn = delivery_sheet_template_fn

    if logo_fn := config.get("logo_fn"):
        setup.logo_fn = logo_fn

    setup.save(only=["delivery_sheet_fn", "logo_fn"])


def migrate(migrator: Migrator, database: pw.Database, *, fake=False):
    """Write your migrations here."""
    
    migrator.add_fields(
        'setup',

        delivery_sheet_fn=pw.TextField(default='delivery_sheet.html'),
        logo_fn=pw.TextField(default='logo.jpg'))

    migrator.run(extract_fns, migrator)


def rollback(migrator: Migrator, database: pw.Database, *, fake=False):
    """Write your rollback migrations here."""
    
    migrator.remove_fields('setup', 'delivery_sheet_fn', "logo_fn")
