"""Peewee migrations -- 148_MAIN.py.

Some examples (model - class or model name)::

    > Model = migrator.orm['table_name']            # Return model in current state by name
    > Model = migrator.ModelClass                   # Return model in current state by name

    > migrator.sql(sql)                             # Run custom SQL
    > migrator.run(func, *args, **kwargs)           # Run python function with the given args
    > migrator.create_model(Model)                  # Create a model (could be used as decorator)
    > migrator.remove_model(model, cascade=True)    # Remove a model
    > migrator.add_fields(model, **fields)          # Add fields to a model
    > migrator.change_fields(model, **fields)       # Change fields
    > migrator.remove_fields(model, *field_names, cascade=True)
    > migrator.rename_field(model, old_field_name, new_field_name)
    > migrator.rename_table(model, new_table_name)
    > migrator.add_index(model, *col_names, unique=False)
    > migrator.add_not_null(model, *field_names)
    > migrator.add_default(model, field_name, default)
    > migrator.add_constraint(model, name, sql)
    > migrator.drop_index(model, *col_names)
    > migrator.drop_not_null(model, *field_names)
    > migrator.drop_constraints(model, *constraints)

"""
from typing import Sequence

import peewee as pw
from peewee_migrate import Migrator


def link_surcharges(migrator: Migrator, *, template_model: str, record_model: str, match_by: Sequence[str]) -> None:
    template_surcharges = {tuple(getattr(s, attribute) for attribute in match_by): s for s in migrator.orm[template_model].select()}

    for surcharge in migrator.orm[record_model].select():
        key = tuple(getattr(surcharge, attribute) for attribute in match_by)

        if template := template_surcharges.get(key):
            surcharge.template_record = template
            surcharge.save(only=["template_record"])


def migrate(migrator: Migrator, database: pw.Database, *, fake=False):
    """Write your migrations here."""
    
    migrator.add_fields(
        'ordersurcharge',

        template_record=pw.ForeignKeyField(column_name='template_record_id', field='id', model=migrator.orm['companysurcharge'], null=True, on_delete='SET NULL'))

    migrator.add_fields(
        'pumpordersurcharge',

        template_record=pw.ForeignKeyField(column_name='template_record_id', field='id', model=migrator.orm['pumpsurcharge'], null=True, on_delete='SET NULL'))

    migrator.run(link_surcharges, migrator, template_model='companysurcharge', record_model='ordersurcharge',
                 match_by=("name", "price", "price_type", "unit_name"))

    migrator.run(link_surcharges, migrator, template_model='pumpsurcharge', record_model='pumpordersurcharge',
                 match_by=("name", "price", "price_type", "unit_name", "export_name"))


def rollback(migrator: Migrator, database: pw.Database, *, fake=False):
    """Write your rollback migrations here."""
    
    migrator.remove_fields('pumpordersurcharge', 'template_record')

    migrator.remove_fields('ordersurcharge', 'template_record')
