"""Peewee migrations -- 162_MAIN.py.

Some examples (model - class or model name)::

    > Model = migrator.orm['table_name']            # Return model in current state by name
    > Model = migrator.ModelClass                   # Return model in current state by name

    > migrator.sql(sql)                             # Run custom SQL
    > migrator.run(func, *args, **kwargs)           # Run python function with the given args
    > migrator.create_model(Model)                  # Create a model (could be used as decorator)
    > migrator.remove_model(model, cascade=True)    # Remove a model
    > migrator.add_fields(model, **fields)          # Add fields to a model
    > migrator.change_fields(model, **fields)       # Change fields
    > migrator.remove_fields(model, *field_names, cascade=True)
    > migrator.rename_field(model, old_field_name, new_field_name)
    > migrator.rename_table(model, new_table_name)
    > migrator.add_index(model, *col_names, unique=False)
    > migrator.add_not_null(model, *field_names)
    > migrator.add_default(model, field_name, default)
    > migrator.add_constraint(model, name, sql)
    > migrator.drop_index(model, *col_names)
    > migrator.drop_not_null(model, *field_names)
    > migrator.drop_constraints(model, *constraints)

"""

import peewee as pw
from peewee_migrate import Migrator


def expand_transport_zone(migrator: Migrator) -> None:
    Delivery = migrator.orm["delivery"]

    for delivery in Delivery.filter(Delivery.transport_zone_record.is_null(False)).iterator():
        zone = delivery.transport_zone_record

        delivery.zone_price_per_m3 = zone.price_per_m3
        delivery.zone_price_is_per_m3 = zone.price_is_per_m3
        delivery.zone_minimal_volume = zone.minimal_volume

        delivery.save(only=["zone_price_per_m3", "zone_price_is_per_m3", "zone_minimal_volume"])


def migrate(migrator: Migrator, database: pw.Database, *, fake=False):
    """Write your migrations here."""

    migrator.remove_fields('delivery', 'price_transport')

    migrator.remove_fields('order', 'without_transport')

    migrator.add_fields(
        'delivery',

        zone_price_is_per_m3=pw.DoubleField(null=True),
        zone_price_per_m3=pw.DoubleField(null=True),
        zone_minimal_volume=pw.DoubleField(null=True))

    migrator.run(expand_transport_zone, migrator)


def rollback(migrator: Migrator, database: pw.Database, *, fake=False):
    """Write your rollback migrations here."""

    migrator.remove_fields('delivery', 'zone_price_is_per_m3', 'zone_price_per_m3', 'zone_minimal_volume')

    migrator.add_fields(
        'order',

        without_transport=pw.BooleanField(default=False))

    migrator.add_fields(
        'delivery',

        price_transport=pw.DoubleField(null=True))
