import { app, BrowserWindow, screen } from 'electron'
import fs from 'fs'
import os from 'os'

import minimist from 'minimist'

import { sendToBottomAndKeepItThere } from './send_to_bottom.js'

const RETRY_DELAY = 3000 // ms

const getArgs = () => {
  const parsed = minimist(process.argv.slice(1))
  return {
    showHelp: parsed.help || parsed.h,
    url: parsed.url || 'http://[::1]:8899',
    border: parsed.border === undefined ? false : parseInt(parsed.border) === 1,
    windowSize: parsed.size ? parsed.size.split('x').map(Number) : undefined,
    quitFilePath: 'quit'
  }
}

const args = getArgs()

// Keep a global reference of the window object, if you don't, the window will
// be closed automatically when the JavaScript object is garbage collected.
let mainWindow
let deinitFileLogging

let quitIntervalId
let retryTimeoutId

function showHelpText () {
  console.log(`
Usage: control_gui4 [options]

Options:
  --help, -h            Show this help text
  --url <URL>           Specify the URL to load (default: http://[::1]:8899)
  --border <0|1>        Set to 0 for borderless window, 1 for window with border (default: 0)
  --size <WIDTHxHEIGHT> Set the initial window size (e.g., --size 1600x900)
`)
  app.exit()
}

if (args.showHelp) {
  showHelpText()
} else {
  deinitFileLogging = initFileLogging()

  console.log('*'.repeat(40))
  console.log(`starting control_gui4 v${app.getVersion()}`)
  console.log(`electron version: ${process.versions.electron}`)
  console.log(`node version: ${process.versions.node}`)
  console.log(`chrome version: ${process.versions.chrome}`)

  if (fs.existsSync(args.quitFilePath)) {
    console.log('stale quit file found, removing it')
    fs.unlinkSync(args.quitFilePath)
  }
}

function formatCurrentTime () {
  const now = new Date()

  const year = now.getFullYear()
  const month = String(now.getMonth() + 1).padStart(2, '0')
  const day = String(now.getDate()).padStart(2, '0')
  const hours = String(now.getHours()).padStart(2, '0')
  const minutes = String(now.getMinutes()).padStart(2, '0')
  const seconds = String(now.getSeconds()).padStart(2, '0')
  const milliseconds = String(now.getMilliseconds()).padStart(3, '0')

  return `${year}-${month}-${day} ${hours}:${minutes}:${seconds}.${milliseconds}`
}

function initFileLogging () {
  let logStream = fs.createWriteStream('/atx300/log/control_gui4.log', { flags: 'a' })

  const fileConsole = consoleMethod => {
    return (message, ...args) => {
      logStream?.write(`${formatCurrentTime()}: ${consoleMethod.name.toUpperCase()}: ${message} ${args}\n`)
      consoleMethod(message, ...args)
    }
  }

  const consoleDebugOriginal = console.debug
  const consoleLogOriginal = console.log
  const consoleErrorOriginal = console.error
  console.debug = fileConsole(console.debug)
  console.log = fileConsole(console.log)
  console.error = fileConsole(console.error)

  return () => {
    console.debug('deinitFileLogging')
    console.debug = consoleDebugOriginal
    console.log = consoleLogOriginal
    console.error = consoleErrorOriginal
    logStream?.close()
    logStream = undefined
  }
}

const windowOptions = () => {
  const options = {
    webPreferences: {
      nodeIntegration: true
    },
    scrollBounce: false,
    autoHideMenuBar: true,
    type: 'desktop', // try to be always-on-bottom - does not really work on windows so we send it to bottom with native call hack
    titleBarStyle: 'hiddenInset' // hide title bar
  }

  if (args.border || args.windowSize?.length > 0) {
    options.width = args.windowSize[0]
    options.height = args.windowSize[1]
  } else {
    const { width, height } = screen.getPrimaryDisplay().workAreaSize
    options.width = width
    // options.fullscreen = true;
    options.height = height - 1
  }
  options.resizable = args.border
  options.frame = args.border

  return options
}

function createWindow () {
  mainWindow = new BrowserWindow(windowOptions())

  if (os.platform() === 'win32') {
    console.log('detected windows platform')
    sendToBottomAndKeepItThere(mainWindow)
  } else {
    console.log('detected non-windows platform')
  }

  // Open the DevTools.
  // mainWindow.webContents.openDevTools()

  // Prevent quit for the first 6 times
  let closeAttempts = 0
  let firstAttemptTime = null
  mainWindow.on('close', e => {
    const now = Date.now()
    if (firstAttemptTime === null || now - firstAttemptTime > 3000) {
      firstAttemptTime = now
      closeAttempts = 1
    } else {
      closeAttempts += 1
    }

    if (mainWindow && closeAttempts < 7) {
      console.log('preventing closing')
      e.preventDefault()
    }
  })

  mainWindow.on('closed', function () {
    // Dereference the window object, usually you would store windows
    // in an array if your app supports multi windows, this is the time
    // when you should delete the corresponding element.
    mainWindow = null
  })
  mainWindow.webContents.on('did-finish-load', () => {
    mainWindow.webContents.executeJavaScript('document.body.innerHTML').then((html) => {
      if (html.toLowerCase().includes('server error')) {
        console.error('Page loaded with server error. Retrying in 3 seconds...')
        // TODO: this is a very disturbing message - maybe just show some spinner or something...
        const errorMessage = `
<pre>
+====+
|(::)|
| )( |
|(..)|
+====+
</pre>
<p>Retrying in 3 seconds...</p>
`
        mainWindow?.webContents.executeJavaScript(`document.body.innerHTML = \`${errorMessage}\`;`).catch(console.error)
        clearTimeout(retryTimeoutId)
        retryTimeoutId = setTimeout(() => {
          mainWindow?.loadURL(args.url)
        }, RETRY_DELAY)
      } else {
        console.log('Page loaded successfully')
      }
    }).catch(console.error)
  })

  mainWindow.webContents.on('did-fail-load', (_event, errorCode, errorDescription, validatedURL, _isMainFrame) => {
    console.error(`Failed to load page: ${validatedURL}, with error: ${errorDescription} (Code: ${errorCode}). Retrying in 3 seconds...`)
    const errorMessage = `<h1>Failed to load</h1><p>Error: ${errorDescription} (Code: ${errorCode})</p><p>Retrying in 3 seconds...</p>`
    mainWindow?.webContents.executeJavaScript(`document.body.innerHTML = \`${errorMessage}\`;`).catch(console.error)
    clearTimeout(retryTimeoutId)
    retryTimeoutId = setTimeout(() => {
      mainWindow?.loadURL(args.url)
    }, RETRY_DELAY)
  })

  mainWindow.webContents.on('before-input-event', (event, input) => {
    if (input.control && input.shift && input.key.toLowerCase() === 'i') {
      mainWindow.webContents.openDevTools()
      event.preventDefault()
    } else if (input.control && input.key.toLowerCase() === 'r') {
      mainWindow.reload()
      event.preventDefault()
    } else if (input.control && input.key.toLowerCase() === 'f') {
      mainWindow.setFullScreen(!mainWindow.isFullScreen())
      event.preventDefault()
    } else if (input.control && input.shift && input.key.toLowerCase() === 'v') { // just for testing - TODO: get rid of this in the future
      const { width, height } = screen.getPrimaryDisplay().workAreaSize
      mainWindow.setSize(width, height - 1)
      mainWindow.setPosition(0, 0)
      event.preventDefault()
    }
  })

  quitIntervalId = setInterval(checkQuitFile, 1000)

  // Disable scrollbars
  mainWindow.webContents.on('dom-ready', () => {
    mainWindow.webContents.insertCSS('body::-webkit-scrollbar { display: none; } body { overflow: hidden !important; margin: 0; }')
  })
  mainWindow.loadURL(args.url)
}

// This method will be called when Electron has finished
// initialization and is ready to create browser windows.
// Some APIs can only be used after this event occurs.
app.whenReady().then(createWindow)

app.on('will-quit', () => {
  // This should be the last event to fire before exit
  // https://github.com/electron/electron/blob/main/docs/api/app.md#event-will-quit
  console.debug('will-quit')
  deinitFileLogging?.()
})

// Quit when all windows are closed, except on macOS. There, it's common
// for applications and their menu bar to stay active until the user quits
// explicitly with Cmd + Q.
app.on('window-all-closed', () => {
  console.debug('window-all-closed')
  if (process.platform !== 'darwin') {
    console.debug('app.quit()')
    app.quit()
  }
})

app.on('activate', () => {
  // On macOS it's common to re-create a window in the app when the
  // dock icon is clicked and there are no other windows open.
  if (BrowserWindow.getAllWindows().length === 0) {
    console.debug('createWindow()')
    createWindow()
  }
})

function cleanUp () {
  console.debug('setting window to null')
  mainWindow = null
  console.debug('will clear intervals and timeouts')
  console.debug(`quitIntervalId: ${quitIntervalId}, retryTimeoutId: ${retryTimeoutId}`)
  clearInterval(quitIntervalId)
  clearTimeout(retryTimeoutId)
}

function checkQuitFile () {
  if (fs.existsSync(args.quitFilePath)) {
    console.log('quit file found, removing it and exiting')
    fs.unlinkSync(args.quitFilePath)
    cleanUp()
    console.debug('will quit app')
    app.quit()
  }
}
