// Implementation of bottommost window based on
// https://www.codeproject.com/tips/348061/creating-a-bottommost-window
// and https://stackoverflow.com/questions/39435066/change-system-z-order-of-browserwindow-in-electron-possible

import koffi from 'koffi'

// Windows constants
const HWND_BOTTOM = 1
const SWP_NOACTIVATE = 0x0010
const SWP_NOMOVE = 0x0002
const SWP_NOSIZE = 0x0001
const SWP_NOSENDCHANGING = 0x0400

// Window messages to intercept
const WM_WINDOWPOSCHANGING = 0x0046

/**
 * Sets the Electron Window to "always-on-bottom" and intercepts windows messages
 * to ensure it stays at the bottom of the z-order
 * @param {Electron.BrowserWindow} window
 */
export const sendToBottomAndKeepItThere = window => {
  // Load Windows API functions
  const user32 = koffi.load('user32.dll')

  const SetWindowPos = user32.func('__stdcall', 'SetWindowPos', 'bool', [
    /* hwnd */ 'size_t',
    /* hwndInsertAfter */ 'int32',
    /* x */ 'int32',
    /* y */ 'int32',
    /* cx */ 'int32',
    /* cy */ 'int32',
    /* flags */ 'uint32'
  ])

  // Track if we're already processing a position change to prevent flicker
  let isChangingPosition = false

  // Send window to bottom initially
  const sendToBottom = win => {
    if (isChangingPosition) return

    try {
      isChangingPosition = true
      const hwnd = win.getNativeWindowHandle().readUInt32LE()
      SetWindowPos(
        hwnd,
        HWND_BOTTOM,
        0, 0, 0, 0,
        SWP_NOACTIVATE | SWP_NOMOVE | SWP_NOSIZE | SWP_NOSENDCHANGING
      )
    } catch (err) {
      console.error('Error sending window to bottom:', err.message)
    } finally {
      isChangingPosition = false
    }
  }

  // Send to bottom on start
  sendToBottom(window)

  // Intercept the window position changing message to ensure window stays at bottom
  window.hookWindowMessage(WM_WINDOWPOSCHANGING, (wParam, lParam) => {
    try {
      if (!lParam || !lParam.buffer) {
        return false
      }

      // Always use offset 4 since we're getting an error with offset 8
      // This indicates we're running in 32-bit mode regardless of process.arch
      const buffer = Buffer.from(lParam.buffer)

      if (buffer.length >= 8) {
        // Safely write to offset 4
        buffer.writeUInt32LE(HWND_BOTTOM, 4)
      } else {
        console.warn('Buffer too small for WINDOWPOS structure:', buffer.length)
        return false
      }

      return true
    } catch (err) {
      // Avoid excessive logging - just log the first few errors
      const errorMsg = `Error in hook: ${err.message}, buffer length: ${lParam && lParam.buffer ? lParam.buffer.length : 'N/A'}`
      console.error(errorMsg)
      return false
    }
  })

  // Handle click/focus without immediate flicker
  let focusTimeout = null
  const handleFocusEvent = () => {
    if (focusTimeout) {
      clearTimeout(focusTimeout)
    }
    focusTimeout = setTimeout(() => {
      sendToBottom(window)
      focusTimeout = null
    }, 100) // Small delay helps prevent flickering
  }

  // Listen for Electron focus event
  window.on('focus', handleFocusEvent)
  window.on('blur', handleFocusEvent) // Also handle blur events

  // Keep sending to bottom periodically without logging
  setInterval(() => {
    if (!window.isDestroyed()) {
      try {
        const hwnd = window.getNativeWindowHandle().readUInt32LE()
        SetWindowPos(
          hwnd,
          HWND_BOTTOM,
          0, 0, 0, 0,
          SWP_NOACTIVATE | SWP_NOMOVE | SWP_NOSIZE | SWP_NOSENDCHANGING
        )
      } catch (err) {
        // Silently ignore interval errors
      }
    }
  }, 2000)
}
